
{-# OPTIONS -fglasgow-exts #-}

import Debug.Trace
import Control.Monad
import Control.Monad.Identity
import Control.Monad.State
import Control.Monad.Writer
import OponOp

-- our arithmetics language

data Arith a = Con Int | Add a a

type ArithVal = Int

instance Functor Arith where
    fmap f (Con x)    = Con x
    fmap f (Add t u) = Add (f t) (f u)

instance Traversable Arith where
    traverse  g (Con x)    = pure $ Con x
    traverse  g (Add t u)  = pure Add <*> g t <*> g u

natopArith  :: Alg Arith ArithVal
natopArith  = Alg natop where
    natop (Con x)    = x
    natop (Add t u)  = t + u

-- our exceptions language

data Exc a      =  Throw | Catch a a
		   deriving Show

instance Functor Exc where
    fmap f Throw        = Throw
    fmap f (Catch a b)  = Catch (f a) (f b)

instance Traversable Exc where
    traverse g Throw        = pure $ Throw
    traverse g (Catch a b)  = pure Catch <*> g a <*> g b

instance Applicative Maybe where
    pure   = return
    (<*>)  = ap

tgermExc                     :: TransGerm Exc Maybe
tgermExc Throw               = Nothing
tgermExc (Catch Nothing u)   = u
tgermExc (Catch (Just x) _)  = Just x

exctrans :: Functor b => OpRule Exc (Comp Maybe b)
exctrans = mkRuleTrans tgermExc

-- a language with state

newtype Tick a = Tick a deriving Show

instance Functor Tick where
    fmap f (Tick a)  = Tick (f a)

instance Traversable Tick where
    traverse g (Tick a)   = pure Tick <*> g a

type MyState = Int

instance Applicative (State b) where
    pure   = return
    (<*>)  = ap

tgermState :: TransGerm Tick (State MyState)
tgermState (Tick t) = do  s <- get
  		          put (s+1)
                          t

statetrans  :: Functor b => OpRule Tick (Comp (State MyState) b)
statetrans  = mkRuleTrans tgermState

-- A trace language

newtype Trace a = Trace (a,String) deriving Show

instance Functor Trace where
    fmap f (Trace (a,o))  = Trace (f a,o)

instance Traversable Trace where
    traverse g (Trace (a,o))   = pure Trace <*> fmap (flip (,) o) (g a)

instance Applicative (Writer String) where
    pure   = return
    (<*>)  = ap

tgermTrace :: TransGerm Trace (Writer String)
tgermTrace (Trace (t,o)) = tell o >> t

tracetrans ::  Functor b => OpRule Trace (Comp (Writer String) b)
tracetrans = mkRuleTrans tgermTrace


-- Some Test programs
run = exec opAE testAE1

testAE1  :: Program (Sum Arith Exc)
testAE1  = mycatch (add (con 3) throw) (add (add (con 2) (con 2)) (con 5))

con          = Term. Inl. Con
add x y      = Term $ Inl $ Add x y
mycatch x y  = Term $ Inr $ Catch x y
throw        = Term $ Inr $ Throw

opAE = joinOS (liftOS (alg2op natopArith)) exctrans

testAS  :: Program (Sum Arith Tick)
testAS  =  tick $ add (con 3) (tick $ add (con 4) (con 5))

tick       = Term. Inr. Tick

opAS = joinOS (liftOS (alg2op natopArith)) statetrans

testAO1  :: Program (Sum Arith Trace)
testAO1 =  traceadd " + " (tracecon 3) 
                          (traceadd " + " (tracecon 4) 
                                          (tracecon 5))
testAO2 = traceadd "a1"  (traceadd "a2"  (con 1) 
                                         (traceadd "a4" (con 4) (con 4))) 
                         (traceadd "a3"  (con 2) (con 6))
			 
mytrace s x  = Term $ Inr $ Trace (x,s)
traceadd s x y  = mytrace s (add x y)
tracecon x      = mytrace (show x) (con x)

opAO  = joinOS (liftOS (alg2op natopArith)) tracetrans
opAEO = joinOS (liftOS opAE) tracetrans

-- Many Show instances

instance Show a => Show (Arith a) where
    show (Con x)    = "Con "++show x
    show (Add x y)  = "Add ("++show x++") ("++show y++")"


instance PreservesShow Arith where
    preservesShow = show

instance PreservesShow Trace where
    preservesShow = show

instance PreservesShow Exc where
    preservesShow = show

instance PreservesShow Maybe where
    preservesShow = show

instance Show a => PreservesShow  (Const a) where
    preservesShow (Const a) = show a

instance Show a => Show (State MyState a) where
    show f = "value: "++show x++", count: "++show s
	     where (x,s) = runState f 0

instance PreservesShow (State MyState) where
    preservesShow f = "value: "++show x++", count: "++show s
	     where (x,s) = runState f 0

instance  (PreservesShow a, Functor a, PreservesShow b) => 
          PreservesShow (Comp a b) where
    preservesShow (Comp a) = preservesShow (fmap preservesShow a)

instance Show a => Show (Writer String a) where
    show w = trace++show a where (a,trace) = runWriter w

instance PreservesShow (Writer String) where
    preservesShow = show

